/*
 *  $Id: volume_facetlevel.c 28905 2025-11-24 15:51:18Z yeti-dn $
 *  Copyright (C) 2018-2024 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <gwy.h>
#include "libgwyddion/omp.h"

#define RUN_MODES (GWY_RUN_IMMEDIATE)

static gboolean module_register(void);
static void     module_main    (GwyFile *data,
                                GwyRunModeFlags mode);
static gboolean brick_level    (GwyBrick *brick,
                                GtkWindow *wait_window);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Performs facet leveling on all XY planes"),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.1",
    "David Nečas (Yeti) & Petr Klapetek",
    "2023",
};

GWY_MODULE_QUERY2(module_info, volume_facetlevel)

static gboolean
module_register(void)
{
    gwy_volume_func_register("volume_facetlevel",
                             module_main,
                             N_("/_Correct Data/_XY Facet Level"),
                             NULL,
                             RUN_MODES,
                             GWY_MENU_FLAG_VOLUME,
                             N_("Facet leveling of all XY planes"));

    return TRUE;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    GwyBrick *brick = NULL;
    gint id, newid;

    g_return_if_fail(mode & RUN_MODES);

    gwy_data_browser_get_current(GWY_APP_BRICK, &brick,
                                 GWY_APP_BRICK_ID, &id,
                                 0);
    g_return_if_fail(GWY_IS_BRICK(brick));

    brick = gwy_brick_copy(brick);
    if (brick_level(brick, gwy_data_browser_get_window_for_data(data, GWY_FILE_VOLUME, id))) {
        newid = gwy_file_add_volume(data, brick, NULL);
        gwy_file_set_visible(data, GWY_FILE_VOLUME, newid, TRUE);
        gwy_file_set_title(data, GWY_FILE_VOLUME, newid, _("Facet leveled"), TRUE);
        gwy_file_sync_items(data, GWY_FILE_VOLUME, id,
                            data, GWY_FILE_VOLUME, newid,
                            GWY_FILE_ITEM_PALETTE, FALSE);
        gwy_log_add(data, GWY_FILE_VOLUME, id, newid);
    }
    g_object_unref(brick);
}

static gboolean
brick_level(GwyBrick *brick, GtkWindow *wait_window)
{
    gint zres = gwy_brick_get_zres(brick);
    gboolean cancelled = FALSE, *pcancelled = &cancelled;

    if (wait_window)
        gwy_app_wait_start(wait_window, _("Facet-leveling..."));

#ifdef _OPENMP
#pragma omp parallel if(gwy_threads_are_enabled()) default(none) \
            shared(brick,zres,wait_window,pcancelled)
#endif
    {
        GwyField *field = gwy_brick_new_field_like_xy_plane(brick, FALSE);
        gint kfrom = gwy_omp_chunk_start(zres), kto = gwy_omp_chunk_end(zres);
        gint k;

        for (k = kfrom; k < kto; k++) {
            gwy_brick_extract_xy_plane(brick, field, k);
            gwy_field_facet_level(field, NULL, GWY_MASK_IGNORE, -1, NULL);
            gwy_brick_set_xy_plane(brick, field, k);

            if (wait_window && gwy_omp_set_fraction_check_cancel(gwy_app_wait_set_fraction, k, kfrom, kto, pcancelled))
                break;
        }

        g_object_unref(field);
    }

    if (wait_window)
        gwy_app_wait_finish();

    return !cancelled;
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
