/*
 * (C) Copyright 1996- ECMWF.
 *
 * This software is licensed under the terms of the Apache Licence Version 2.0
 * which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
 *
 * In applying this licence, ECMWF does not waive the privileges and immunities
 * granted to it by virtue of its status as an intergovernmental organisation nor
 * does it submit to any jurisdiction.
 */


#include <memory>
#include <vector>

#include "eckit/geo/Grid.h"
#include "eckit/geo/Point.h"
#include "eckit/spec/Custom.h"
#include "eckit/testing/Test.h"


namespace eckit::geo::test {


CASE("points (global)") {
    spec::Custom spec{{"type", "reduced_ll"}, {"pl", std::vector<long>{20, 24, 24, 20}}};
    std::unique_ptr<const Grid> grid(GridFactory::build(spec));

    EXPECT_EQUAL(grid->size(), 88);

    const std::vector<PointLonLat> ref{
        {0., 90.},    {18., 90.},   {36., 90.},   {54., 90.},   {72., 90.},    //
        {90., 90.},   {108., 90.},  {126., 90.},  {144., 90.},  {162., 90.},   //
        {180., 90.},  {198., 90.},  {216., 90.},  {234., 90.},  {252., 90.},   //
        {270., 90.},  {288., 90.},  {306., 90.},  {324., 90.},  {342., 90.},   //
        {0., 30.},    {15., 30.},   {30., 30.},   {45., 30.},   {60., 30.},    //
        {75., 30.},   {90., 30.},   {105., 30.},  {120., 30.},  {135., 30.},   //
        {150., 30.},  {165., 30.},  {180., 30.},  {195., 30.},  {210., 30.},   //
        {225., 30.},  {240., 30.},  {255., 30.},  {270., 30.},  {285., 30.},   //
        {300., 30.},  {315., 30.},  {330., 30.},  {345., 30.},                 //
        {0., -30.},   {15., -30.},  {30., -30.},  {45., -30.},  {60., -30.},   //
        {75., -30.},  {90., -30.},  {105., -30.}, {120., -30.}, {135., -30.},  //
        {150., -30.}, {165., -30.}, {180., -30.}, {195., -30.}, {210., -30.},  //
        {225., -30.}, {240., -30.}, {255., -30.}, {270., -30.}, {285., -30.},  //
        {300., -30.}, {315., -30.}, {330., -30.}, {345., -30.},                //
        {0., -90.},   {18., -90.},  {36., -90.},  {54., -90.},  {72., -90.},   //
        {90., -90.},  {108., -90.}, {126., -90.}, {144., -90.}, {162., -90.},  //
        {180., -90.}, {198., -90.}, {216., -90.}, {234., -90.}, {252., -90.},
        {270., -90.}, {288., -90.}, {306., -90.}, {324., -90.}, {342., -90.},
    };

    auto points = grid->to_points();

    EXPECT(points.size() == grid->size());
    ASSERT(points.size() == ref.size());

    auto it = grid->begin();
    for (size_t i = 0; i < points.size(); ++i) {
        EXPECT(points_equal(ref[i], points[i]));
        EXPECT(points_equal(ref[i], *it));
        ++it;
    }
    EXPECT(it == grid->end());

    size_t i = 0;
    for (const auto& it : *grid) {
        EXPECT(points_equal(ref[i++], it));
    }
    EXPECT(i == grid->size());
}


CASE("points (regional)") {
    spec::Custom spec{
        {"type", "reduced_ll"},  //
        {"pl",
         std::vector<long>{
             2,    4,    6,    8,    12,   14,   16,   20,   22,   24,   28,   30,   32,   36,   38,   40,   42,   46,
             48,   50,   54,   56,   58,   62,   64,   66,   70,   72,   74,   78,   80,   82,   86,   88,   90,   94,
             96,   98,   102,  104,  106,  110,  112,  114,  118,  120,  122,  126,  128,  130,  134,  136,  138,  140,
             144,  146,  148,  152,  154,  156,  160,  162,  164,  168,  170,  172,  176,  178,  180,  184,  186,  188,
             192,  194,  196,  200,  202,  204,  206,  210,  212,  214,  218,  220,  222,  226,  228,  230,  234,  236,
             238,  242,  244,  246,  248,  252,  254,  256,  260,  262,  264,  268,  270,  272,  276,  278,  280,  282,
             286,  288,  290,  294,  296,  298,  302,  304,  306,  308,  312,  314,  316,  320,  322,  324,  328,  330,
             332,  334,  338,  340,  342,  346,  348,  350,  352,  356,  358,  360,  364,  366,  368,  372,  374,  376,
             378,  382,  384,  386,  390,  392,  394,  396,  400,  402,  404,  406,  410,  412,  414,  418,  420,  422,
             424,  428,  430,  432,  436,  438,  440,  442,  446,  448,  450,  452,  456,  458,  460,  462,  466,  468,
             470,  474,  476,  478,  480,  484,  486,  488,  490,  494,  496,  498,  500,  504,  506,  508,  510,  514,
             516,  518,  520,  524,  526,  528,  530,  534,  536,  538,  540,  544,  546,  548,  550,  554,  556,  558,
             560,  562,  566,  568,  570,  572,  576,  578,  580,  582,  586,  588,  590,  592,  594,  598,  600,  602,
             604,  606,  610,  612,  614,  616,  620,  622,  624,  626,  628,  632,  634,  636,  638,  640,  644,  646,
             648,  650,  652,  656,  658,  660,  662,  664,  668,  670,  672,  674,  676,  680,  682,  684,  686,  688,
             692,  694,  696,  698,  700,  702,  706,  708,  710,  712,  714,  716,  720,  722,  724,  726,  728,  730,
             734,  736,  738,  740,  742,  744,  748,  750,  752,  754,  756,  758,  762,  764,  766,  768,  770,  772,
             774,  778,  780,  782,  784,  786,  788,  790,  792,  796,  798,  800,  802,  804,  806,  808,  812,  814,
             816,  818,  820,  822,  824,  826,  828,  832,  834,  836,  838,  840,  842,  844,  846,  848,  852,  854,
             856,  858,  860,  862,  864,  866,  868,  870,  872,  876,  878,  880,  882,  884,  886,  888,  890,  892,
             894,  896,  898,  900,  904,  906,  908,  910,  912,  914,  916,  918,  920,  922,  924,  926,  928,  930,
             932,  934,  936,  940,  942,  944,  946,  948,  950,  952,  954,  956,  958,  960,  962,  964,  966,  968,
             970,  972,  974,  976,  978,  980,  982,  984,  986,  988,  990,  992,  994,  996,  998,  1000, 1002, 1004,
             1006, 1008, 1010, 1012, 1014, 1016, 1018, 1020, 1022, 1024, 1026, 1028, 1030, 1032, 1034, 1036, 1038, 1040,
             1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1058, 1060, 1062, 1064, 1066, 1068, 1070, 1072, 1074,
             1076, 1078, 1080, 1082, 1084, 1086, 1088, 1090, 1090, 1092, 1094, 1096, 1098, 1100, 1102, 1104, 1106, 1108,
             1110, 1112, 1112, 1114, 1116, 1118, 1120, 1122, 1124, 1126, 1128, 1130, 1130, 1132, 1134, 1136, 1138, 1140,
             1142, 1144, 1144, 1146, 1148, 1150, 1152, 1154, 1156, 1158, 1158, 1160, 1162, 1164, 1166, 1168, 1170, 1170,
             1172, 1174, 1176, 1178, 1180, 1180, 1182, 1184, 1186, 1188, 1190, 1190, 1192, 1194, 1196, 1198, 1200, 1200,
             1202, 1204, 1206, 1208, 1208, 1210, 1212, 1214, 1216, 1216, 1218, 1220, 1222, 1224, 1224, 1226, 1228, 1230,
             1232, 1232, 1234, 1236, 1238, 1238, 1240, 1242, 1244, 1244, 1246, 1248, 1250, 1250, 1252, 1254, 1256, 1256,
             1258, 1260, 1262, 1262, 1264, 1266, 1268, 1268, 1270, 1272, 1274, 1274, 1276, 1278, 1280, 1280, 1282, 1284,
             1284, 1286, 1288, 1288, 1290, 1292, 1294, 1294, 1296, 1298, 1298, 1300, 1302, 1302, 1304, 1306, 1306, 1308,
             1310, 1312, 1312, 1314, 1316, 1316, 1318, 1320, 1320, 1322, 1324, 1324, 1326, 1326, 1328, 1330, 1330, 1332,
             1334, 1334, 1336, 1338, 1338, 1340, 1340, 1342, 1344, 1344, 1346, 1348, 1348, 1350, 1350, 1352, 1354, 1354,
             1356, 1356, 1358, 1360, 1360, 1362, 1362, 1364, 1366, 1366, 1368, 1368, 1370, 1372, 1372, 1374, 1374, 1376,
             1376, 1378, 1378, 1380, 1382, 1382, 1384, 1384, 1386, 1386, 1388, 1388, 1390, 1392, 1392, 1394, 1394, 1396,
             1396, 1398, 1398, 1400, 1400, 1402, 1402, 1404, 1404, 1406, 1406, 1408, 1408, 1410, 1410, 1412, 1412, 1414,
             1414, 1416, 1416, 1418, 1418, 1420, 1420, 1422, 1422, 1424, 1424, 1426, 1426, 1428, 1428, 1428, 1430, 1430,
             1432, 1432, 1434, 1434, 1436, 1436, 1436, 1438, 1438, 1440, 1440, 1442, 1442, 1442, 1444, 1444, 1446, 1446,
             1448, 1448, 1448, 1450, 1450, 1452, 1452, 1452, 1454, 1454, 1456, 1456, 1456, 1458, 1458, 1458, 1460, 1460,
             1462, 1462, 1462, 1464, 1464, 1464, 1466, 1466, 1466, 1468, 1468, 1468, 1470, 1470, 1472, 1472, 1472, 1474,
             1474, 1474, 1476, 1476, 1476, 1476, 1478, 1478, 1478, 1480, 1480, 1480, 1482, 1482, 1482, 1484, 1484, 1484,
             1484, 1486, 1486, 1486, 1488, 1488, 1488, 1488, 1490, 1490, 1490, 1490, 1492, 1492, 1492, 1494, 1494, 1494,
             1494, 1496, 1496, 1496, 1496, 1496, 1498, 1498, 1498, 1498, 1500, 1500, 1500, 1500, 1502, 1502, 1502, 1502,
             1502, 1504, 1504, 1504, 1504, 1504, 1506, 1506, 1506, 1506, 1506, 1508, 1508, 1508, 1508, 1508, 1508, 1510,
             1510, 1510, 1510, 1510, 1510, 1512, 1512, 1512, 1512, 1512, 1512, 1512, 1514, 1514, 1514, 1514, 1514, 1514,
             1514, 1514, 1516, 1516, 1516,
         }},                                                //
        {"area", std::vector<double>{90., -98., 5., 54.}},  //
    };

    const std::vector<PointLonLat> ref{
        {-98., 90.},           {54., 90.},
        {-98., 89.9},          {-47.333333333, 89.9},
        {3.333333333, 89.9},   {54., 89.9},
        {-98., 89.8},          {-67.6, 89.8},
        {-37.2, 89.8},         {-6.8, 89.8},
        {23.6, 89.8},          {54., 89.8},
        {-98., 89.7},          {-76.285714286, 89.7},
        {-54.571428571, 89.7}, {-32.857142857, 89.7},
        {-11.142857143, 89.7}, {10.571428571, 89.7},
        {32.285714286, 89.7},  {54., 89.7},
        {-98., 89.6},          {-84.181818182, 89.6},
        {-70.363636364, 89.6}, {-56.545454545, 89.6},
        {-42.727272727, 89.6}, {-28.909090909, 89.6},
        {-15.090909091, 89.6}, {-1.272727273, 89.6},
        {12.545454545, 89.6},  {26.363636364, 89.6},
        {40.181818182, 89.6},  {54., 89.6},
        {-98., 89.5},          {-86.307692308, 89.5},
        {-74.615384615, 89.5}, {-62.923076923, 89.5},
        {-51.230769231, 89.5}, {-39.538461538, 89.5},
        {-27.846153846, 89.5}, {-16.153846154, 89.5},
        {-4.461538462, 89.5},  {7.230769231, 89.5},
        {18.923076923, 89.5},  {30.615384615, 89.5},
        {42.307692308, 89.5},  {54., 89.5},
        {-98., 89.4},          {-87.866666667, 89.4},
        {-77.733333333, 89.4}, {-67.6, 89.4},
        {-57.466666667, 89.4}, {-47.333333333, 89.4},
        {-37.2, 89.4},         {-27.066666667, 89.4},
        {-16.933333333, 89.4}, {-6.8, 89.4},
        {3.333333333, 89.4},   {13.466666667, 89.4},
        {23.6, 89.4},          {33.733333333, 89.4},
        {43.866666667, 89.4},  {54., 89.4},
        {-98., 89.3},          {-90., 89.3},
        {-82., 89.3},          {-74., 89.3},
        {-66., 89.3},          {-58., 89.3},
        {-50., 89.3},
    };

    constexpr size_t EXPECTED_SIZE = 796702;

    std::unique_ptr<const Grid> grid(GridFactory::build(spec));
    auto points = grid->to_points();

    EXPECT(grid->size() == EXPECTED_SIZE);
    EXPECT(points.size() == EXPECTED_SIZE);

    auto it = grid->begin();
    for (size_t i = 0; i < ref.size(); ++i) {
        EXPECT(points_equal(ref[i], points[i]));
        EXPECT(points_equal(ref[i], *it));
        ++it;
    }
}


CASE("points (contains zeros)") {
    spec::Custom spec{
        {"type", "reduced_ll"},  //
        {"pl",
         std::vector<long>{
             0,    0,    0,    0,    0,    0,    0,    0,   0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
             0,    0,    0,    0,    0,    0,    0,    156, 164,  170,  176,  182,  188,  194,  200,  206,  212,  218,
             224,  230,  236,  244,  250,  256,  262,  268, 274,  280,  286,  292,  298,  304,  310,  316,  322,  328,
             334,  340,  346,  352,  356,  362,  368,  374, 380,  386,  392,  398,  404,  410,  414,  420,  426,  432,
             438,  444,  448,  454,  460,  466,  472,  476, 482,  488,  494,  498,  504,  510,  514,  520,  526,  532,
             536,  542,  546,  552,  558,  562,  568,  572, 578,  584,  588,  594,  598,  604,  608,  614,  618,  624,
             628,  634,  638,  642,  648,  652,  658,  662, 666,  672,  676,  680,  686,  690,  694,  698,  704,  708,
             712,  716,  720,  726,  730,  734,  738,  742, 746,  750,  754,  758,  762,  766,  772,  776,  778,  782,
             786,  790,  794,  798,  802,  806,  810,  814, 816,  820,  824,  828,  832,  834,  838,  842,  844,  848,
             852,  854,  858,  862,  864,  868,  870,  874, 876,  880,  882,  886,  888,  892,  894,  898,  900,  902,
             906,  908,  910,  914,  916,  918,  920,  924, 926,  928,  930,  932,  934,  938,  940,  942,  944,  946,
             948,  950,  952,  954,  956,  958,  960,  960, 962,  964,  966,  968,  970,  970,  972,  974,  976,  976,
             978,  980,  980,  982,  982,  984,  986,  986, 988,  988,  990,  990,  990,  992,  992,  994,  994,  994,
             996,  996,  996,  998,  998,  998,  998,  998, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000,
             1000, 1000, 1000, 1000, 1000, 1000, 1000, 998, 998,  998,  998,  998,  996,  996,  996,  994,  994,  994,
             992,  992,  990,  990,  990,  988,  988,  986, 986,  984,  982,  982,  980,  980,  978,  976,  976,  974,
             972,  970,  970,  968,  966,  964,  962,  960, 960,  958,  956,  954,  952,  950,  948,  946,  944,  942,
             940,  938,  934,  932,  930,  928,  926,  924, 920,  918,  916,  914,  910,  908,  906,  902,  900,  898,
             894,  892,  888,  886,  882,  880,  876,  874, 870,  868,  864,  862,  858,  854,  852,  848,  844,  842,
             838,  834,  832,  828,  824,  820,  816,  814, 810,  806,  802,  798,  794,  790,  786,  782,  778,  776,
             772,  766,  762,  758,  754,  750,  746,  742, 738,  734,  730,  726,  720,  716,  712,  708,  704,  698,
             694,  690,  686,  680,  676,  672,  666,  662, 658,  652,  648,  642,  638,  634,  628,  624,  618,  614,
             608,  604,  598,  594,  588,  584,  578,  572, 568,  562,  558,  552,  546,  542,  536,  532,  526,  520,
             514,  510,  504,  498,  494,  488,  482,  476, 472,  466,  460,  454,  448,  444,  438,  432,  426,  420,
             414,  410,  404,  398,  392,  386,  380,  374, 368,  362,  356,  352,  346,  340,  334,  328,  322,  316,
             310,  304,  298,  292,  286,  280,  274,  268, 262,  256,  250,  244,  236,  230,  224,  218,  212,  206,
             0,    0,    0,    0,    0,    0,    0,    0,   0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
             0,    0,    0,    0,    0,    0,    0,    0,   0,    0,    0,    0,    0,    0,    0}},
        {"area", std::vector<double>{90., 0., -90., 359.64}},
    };

    constexpr size_t EXPECTED_SIZE = 313362;

    std::unique_ptr<const Grid> grid(GridFactory::build(spec));
    auto points = grid->to_points();

    EXPECT(grid->boundingBox().global());
    EXPECT(grid->size() == EXPECTED_SIZE);
    EXPECT(points.size() == EXPECTED_SIZE);

    EXPECT(points_equal(points[0], PointLonLat{0., 81.}));
    EXPECT(points_equal(points[1], PointLonLat{2.3076923077, 81.}));
    EXPECT(points_equal(points[78], PointLonLat{180., 81.}));
    EXPECT(points_equal(points[79], PointLonLat{182.3076923077, 81.}));
    EXPECT(points_equal(points[156], PointLonLat{0., 80.64}));
    EXPECT(points_equal(points[157], PointLonLat{2.1951219512, 80.64}));
    EXPECT(points_equal(points[238], PointLonLat{180., 80.64}));
    EXPECT(points_equal(points[239], PointLonLat{182.1951219512, 80.64}));
    EXPECT(points_equal(points[320], PointLonLat{0., 80.28}));
    EXPECT(points_equal(points[321], PointLonLat{2.1176470588, 80.28}));
}


}  // namespace eckit::geo::test


int main(int argc, char** argv) {
    return eckit::testing::run_tests(argc, argv);
}
